//
//	GSSerialWin32.cpp
//	 1998, 1999, 2000 Kyle Hammond
//	hammo009@tc.umn.edu
//	Use at your own risk.  All rights reserved.  Do not distribute without permission.
//

#include "GSSerialWin32.h"

class GSSerialWin32AsyncHelper {
	// This class serves to keep the buffers around until the end of a ReadFileEx or WriteFileEx call.
	// It is also used to implement a continous ReadFileEx call on the serial port; in the read completion routine,
	//   the launch function is called again.
public:
	GSSerialWin32AsyncHelper( const HANDLE inDeviceHandle, const unsigned long inBufferSize,
				const bool inDeleteUponCompletion );

	// Call ReadFileEx if inOutboundBuffer == NULL; otherwise call WriteFileEx.
	DWORD launch( const void *inOutboundBuffer, GSSerialWin32 *inComm );

	// finished( ) will return true when the asynchronous I/O is complete
	bool finished( void ) const {	return mFinished;	}

	// Only use these if finished is true.
	void *buffer( void ) const
	{	if ( mFinished )	return mBuffer;
		else			return NULL;
	}
	unsigned long bufferSize( void ) const
	{	if ( mFinished )		return mBytesTransferred;
		else				return 0;
	}
	DWORD completionError( void ) const {	return mFinishedErrorCode;	}

	// Destructor.
	~GSSerialWin32AsyncHelper( );

// Implementation section.
private:
	struct AsyncSerialOverlap {
		// Tack on a pointer to this object at the end of an _OVERLAPPED.
		_OVERLAPPED				overlap;
		GSSerialWin32AsyncHelper	*serialObject;
	} mData;

	HANDLE			mDeviceHandle;
	bool				mFinished, mDeleteUponCompletion;
	unsigned long		mBufferSize;
	char				*mBuffer;
	DWORD			mConstructorError, mFinishedErrorCode, mBytesTransferred;
	GSSerialWin32		*mComm;

	static VOID CALLBACK _writeCompletionRoutine( DWORD dwErrorCode, DWORD dwNumberOFBytesTransferred,
				LPOVERLAPPED lpOverlapped );
	static VOID CALLBACK _readCompletionRoutine( DWORD dwErrorCode, DWORD dwNumberOFBytesTransferred,
				LPOVERLAPPED lpOverlapped );
};

GSSerialWin32AsyncHelper::GSSerialWin32AsyncHelper( const HANDLE inDeviceHandle, const unsigned long inBufferSize,
				const bool inDeleteUponCompletion ) : mDeviceHandle( inDeviceHandle ), mFinished( true ),
				mDeleteUponCompletion( inDeleteUponCompletion ), mBufferSize( inBufferSize ), mBuffer( NULL ),
				mConstructorError( ERROR_SUCCESS ), mFinishedErrorCode( ERROR_SUCCESS ), mBytesTransferred( 0 ),
				mComm( NULL )
{	// Could stash the object pointer in the hEvent field, but then we're more dependent on MS not changing.
	mData.overlap.Offset		= 0;
	mData.overlap.OffsetHigh		= 0;
	mData.overlap.hEvent		= NULL;
	mData.serialObject			= this;

	// Allocate a buffer of the appropriate size.
	mBuffer = new char[ mBufferSize ];

	if ( mBuffer == NULL )
		mConstructorError = GetLastError( );
}

	// Call ReadFileEx if inOutboundBuffer == NULL; otherwise call WriteFileEx.
unsigned long GSSerialWin32AsyncHelper::launch( const void *inOutboundBuffer, GSSerialWin32 *inComm )
{	DWORD	errCode = ERROR_SUCCESS;

	// Report the MemError( ) if the buffer could not be created.
	if ( mConstructorError != ERROR_SUCCESS )
		return mConstructorError;

	mComm = inComm;

	if ( inOutboundBuffer == NULL ) {
		if ( !ReadFileEx( mDeviceHandle, mBuffer, mBufferSize, reinterpret_cast<_OVERLAPPED *>( &mData ),
					_readCompletionRoutine ) )
			errCode = GetLastError( );
		else
			mFinished = false;
	} else {
		// Copy the outgoing data to the local buffer for writing.
		CopyMemory( mBuffer, inOutboundBuffer, mBufferSize );

		if ( !WriteFileEx( mDeviceHandle, mBuffer, mBufferSize, reinterpret_cast<_OVERLAPPED *>( &mData ),
					_writeCompletionRoutine ) )
			errCode = GetLastError( );
		else
			mFinished = false;
	}

	return errCode;
}

GSSerialWin32AsyncHelper::~GSSerialWin32AsyncHelper( )
{	if ( mBuffer != NULL )
		delete [ ] mBuffer;
}

VOID CALLBACK GSSerialWin32AsyncHelper::_writeCompletionRoutine( DWORD dwErrorCode, DWORD dwNumberOFBytesTransferred,
				LPOVERLAPPED lpOverlapped )
{	GSSerialWin32AsyncHelper	*object = reinterpret_cast<AsyncSerialOverlap *>( lpOverlapped )->serialObject;

	// We are done!
	object->mFinished = true;
	object->mFinishedErrorCode = dwErrorCode;
	object->mBytesTransferred = dwNumberOFBytesTransferred;

	// Go ahead and delete the object, if it was a fire-and-forget write.
	if ( object->mDeleteUponCompletion )
		delete object;
}

VOID CALLBACK GSSerialWin32AsyncHelper::_readCompletionRoutine( DWORD dwErrorCode, DWORD dwNumberOFBytesTransferred,
				LPOVERLAPPED lpOverlapped )
{	GSSerialWin32AsyncHelper	*object = reinterpret_cast<AsyncSerialOverlap *>( lpOverlapped )->serialObject;
	unsigned long				index;

	// We are done!
	object->mFinished = true;
	object->mFinishedErrorCode = dwErrorCode;
	object->mBytesTransferred = dwNumberOFBytesTransferred;

	// Place the data into the buffer at the end.
	// Note that excess characters are simply dropped.
	// This is probably OK for Videodisc players, since they don't usually send back much data.
	for ( index = 0; index < dwNumberOFBytesTransferred &&
					object->mComm->mCharsInBuffer < kGSSerialWin32BufferSize; ++index )
		object->mComm->mIncomingBuffer[ object->mComm->mCharsInBuffer++ ] =
					reinterpret_cast<char *>(object->mBuffer)[ index ];


	if ( object->mComm->mContinuousRead )
		// Launch the GSSerialWin32AsyncHelper object again.
		(void)object->launch( NULL, object->mComm );
	else {
		// Go ahead and delete the object, if it was a fire-and-forget read.
		if ( object->mDeleteUponCompletion )
			delete object;
	}
}

#pragma mark -

GSSerialWin32::GSSerialWin32( GSSerialPortInfoRecord *inPortInfo, GSXSerialPortNameRecord *inPortName, const bool inAsynchronous ) :
			GSXSerialCommunicator( inPortInfo, inPortName, inAsynchronous ), mDeviceHandle( INVALID_HANDLE_VALUE )
{ }

unsigned long GSSerialWin32::InitSerialComm( void )
{	DWORD	errCode;
	DWORD	flagsAndAttributes = FILE_ATTRIBUTE_NORMAL;
	char		deviceName[ kWin32PortNameLength + 4 ] = "\\\\.\\";

	mContinuousRead = true;
	mCharsInBuffer = 0;

	if ( mAsync )
		flagsAndAttributes |= FILE_FLAG_OVERLAPPED;

	// For devices, preface with \\.\ and the following parameter values must be used:
	// fdwShareMode == 0, fdwCreate == OPEN_EXISTING, hTemplateFile == NULL

	// Tack on the serial port name to the end of the device preface.
	strcpy( &deviceName[ 4 ], mCurrentPortName.inPortName );

	mDeviceHandle = CreateFile( deviceName, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
				flagsAndAttributes, NULL );

	if ( mDeviceHandle == INVALID_HANDLE_VALUE )
		errCode = GetLastError( );
	else
		errCode = this->SetPortOptions( &mCurrentPortInfo, &mCurrentPortName );

	if ( errCode == ERROR_SUCCESS && mAsync ) {
		// Set up for the asynchronous read.
		// Essentially, we're calling ReadFileEX all the time, waiting for data.
		// The GSSerialWin32AsyncHelper object puts all the data into mIncomingBuffer.
		// Fire and forget about it (auto deleted in call to CloseSerialComm).
		GSSerialWin32AsyncHelper	*helper = new GSSerialWin32AsyncHelper( mDeviceHandle, 1, true );

		errCode = helper->launch( NULL, this );

		if ( errCode != ERROR_SUCCESS )
			delete helper;
	}

	return errCode;
}

unsigned long GSSerialWin32::CloseSerialComm( void )
{	DWORD	errCode = ERROR_SUCCESS;

	mContinuousRead = false;
	if ( mDeviceHandle != INVALID_HANDLE_VALUE ) {
		if ( !CancelIo( mDeviceHandle ) )
			errCode = GetLastError( );

		if ( errCode == ERROR_SUCCESS ) {
			if ( !CloseHandle( mDeviceHandle ) )
				errCode = GetLastError( );
			else
				mDeviceHandle = INVALID_HANDLE_VALUE;
		}
	}

	return errCode;
}

unsigned long GSSerialWin32::SetPortOptions( const GSSerialPortInfoRecord *inPortInfo, const GSXSerialPortNameRecord *inPortName )
{	DCB		deviceState;
	DWORD	errCode = ERROR_SUCCESS;

	if ( !PortRecordsEqual( inPortName, &mCurrentPortName ) ) {
		// Need to switch the port.
		errCode = CloseSerialComm( );
		if ( errCode == ERROR_SUCCESS ) {
			mCurrentPortInfo = *inPortInfo;
			mCurrentPortName = *inPortName;
			errCode = InitSerialComm( );
		}

		return errCode;
	}

	if ( !GetCommState( mDeviceHandle, &deviceState ) )
		errCode = GetLastError( );

	if ( errCode == ERROR_SUCCESS ) {
		// Modify deviceState for new info.

		deviceState.BaudRate = inPortInfo->baud;

		deviceState.ByteSize = inPortInfo->dataBits;

		switch ( inPortInfo->parity ) {
		case kGSSerialParityOdd:		deviceState.Parity = 1;		break;
		case kGSSerialParityEven:	deviceState.Parity = 2;		break;
		case kGSSerialParityNone:
		default:					deviceState.Parity = 0;		break;
		}

		switch ( inPortInfo->stopBits ) {
		case 	kGSSerialStopBits15:	deviceState.StopBits = 1;	break;
		case kGSSerialStopBits20:	deviceState.StopBits = 2;	break;
		case kGSSerialStopBits10:
		default:					deviceState.StopBits = 0;	break;
		}

		// Set the state and get any errors.
		if ( !SetCommState( mDeviceHandle, &deviceState ) )
			errCode = GetLastError( );
	}

	if ( errCode == ERROR_SUCCESS )
		mCurrentPortInfo = *inPortInfo;

	return errCode;
}

unsigned long GSSerialWin32::SendSerialInfo( const void *inBuffer, const unsigned long inBufferSize )
{	DWORD		errCode = ERROR_SUCCESS;
	DWORD		bytesWritten, commErrors;

	if ( mDeviceHandle == INVALID_HANDLE_VALUE )
		return ERROR_INVALID_HANDLE;

	if ( mAsync ) {
		// Fire and forget about it (auto deleted upon completion).
		GSSerialWin32AsyncHelper	*helper = new GSSerialWin32AsyncHelper( mDeviceHandle, inBufferSize, true );

		errCode = helper->launch( inBuffer, this );
		if ( errCode != ERROR_SUCCESS )
			// Error!	Trash the helper.
			delete helper;
	} else {
		if ( !WriteFile( mDeviceHandle, inBuffer, inBufferSize, &bytesWritten, NULL ) )
			errCode = GetLastError( );
	}

	if ( errCode != ERROR_SUCCESS ) {
		if ( !ClearCommError( mDeviceHandle, &commErrors, NULL ) )
			errCode = GetLastError( );
		else
			errCode = commErrors;
	}

	return errCode;
}

unsigned long GSSerialWin32::GetSerialInfo( void *outBuffer, unsigned long *ioCount, const long inWaitTicks )
{
#pragma unused( inWaitTicks )
	DWORD		errCode = ERROR_SUCCESS, commErrors;

	if ( mDeviceHandle == INVALID_HANDLE_VALUE )
		return ERROR_INVALID_HANDLE;

	if ( mAsync ) {
		unsigned long	index;

		// Wait for some data.  How do I do that on Win32?

		// Return a no data error.
		if ( mCharsInBuffer == 0 )
			errCode = ERROR_NO_DATA;

		if ( errCode == ERROR_SUCCESS ) {
			// Copy the characters out of the incoming buffer.
			for ( index = 0; index < *ioCount && index < mCharsInBuffer; index++ )
				reinterpret_cast<char *>( outBuffer )[ index ] = mIncomingBuffer[ index ];

			// This is how many characters we've copied.
			*ioCount = index;
			mCharsInBuffer -= index;

			// Now move the characters in the buffer forward that many characters.
			for ( index = 0; index < mCharsInBuffer; index++ )
				mIncomingBuffer[ index ] = mIncomingBuffer[ index + (*ioCount) ];
		}
	} else {
		if ( !ReadFile( mDeviceHandle, outBuffer, *ioCount, ioCount, NULL ) )
			errCode = GetLastError( );
	}

	if ( errCode != ERROR_SUCCESS ) {
		if ( !ClearCommError( mDeviceHandle, &commErrors, NULL ) )
			errCode = GetLastError( );
	}

	return errCode;
}

static GSXSerialPortNameRecord *_gFakeMasterPtr;

short GSSerialWin32::FindReadySerialPorts( GSXSerialPortNameRecordHandle *outPortInfo, const bool inOpenAndClose )
{
#pragma unused( inOpenAndClose )
	DWORD	resVal;
	HKEY		hSerialCommKey = NULL;
	DWORD	i, status;
	DWORD	keyType, nameBufSize, dataBufSize;
	UCHAR	valueKeyName[ kWin32PortNameLength ], valueKeyData[ kWin32PortNameLength ];

	// First - Get access to the registry key we need

	// We simulate a double-indirect pointer for platform compatibility
	
	_gFakeMasterPtr = NULL;
	*outPortInfo = &_gFakeMasterPtr;
	
	resVal = RegOpenKeyEx( HKEY_LOCAL_MACHINE, "HARDWARE\\DEVICEMAP\\SERIALCOMM",
				0, KEY_READ, &hSerialCommKey );

	// If successful, read all the registry entries under this key

	if ( resVal == ERROR_SUCCESS ) {
		for( i = 0; ; i++ ) {
			nameBufSize = dataBufSize = kWin32PortNameLength;
			status = RegEnumValue( hSerialCommKey, i, (CHAR *)valueKeyName, &nameBufSize,
						NULL, &keyType, (UCHAR *)valueKeyData, &dataBufSize );

			// If we got something, the value of the key is the COM name
			//   otherwise we've come to the end of the list.

			if ( status == ERROR_SUCCESS ) {
				if (_gFakeMasterPtr)
					_gFakeMasterPtr = reinterpret_cast<GSXSerialPortNameRecord *>( realloc( _gFakeMasterPtr,
								( i + 1 ) * sizeof(GSXSerialPortNameRecord) ) );
				else
					_gFakeMasterPtr = reinterpret_cast<GSXSerialPortNameRecord *>( malloc( sizeof(GSXSerialPortNameRecord) ) );
				strncpy( _gFakeMasterPtr[ i ].userPortName, (const char *)valueKeyData, kWin32PortNameLength );
				strncpy( _gFakeMasterPtr[ i ].inPortName, (const char *)valueKeyData, kWin32PortNameLength );
				strncpy( _gFakeMasterPtr[ i ].outPortName, (const char *)valueKeyData, kWin32PortNameLength );
			} else
				break;
		}
		
		// Done, close down the registry before returning

		RegCloseKey( hSerialCommKey );

		// And return the number we found

		return i;
	} else
		return -1;    // Return -1 as an error if we can't access the registry
  }
